*! version 3.1.6  13jan2026 by alexis.dinno@pdx.edu
*! perform two one-sided t tests for mean equivalence

* Syntax:  tostti #obs1 #mean #sd1 #obs2 #mean2 #sd2 [, eqvtype(type) 
*          eqvlevel(#) uppereqvlevel(#) unequal welch alpha(#) xname(string) 
*          yname(string) relevance]
 
 
program define tostti, rclass byable(recall)

  if int(_caller())<8 {
    di in r "tostti- does not support this version of Stata." _newline
    di as txt "All requests for specific version support are welcome and will be considered."
    exit
  }

  if int(_caller())<14 {
    version 8.0, missing
    * Create output constants for non-Unicode versions of Stata
    local alpha_out       = "alpha"
    local theta           = "d"
    local theta_inflate   = "{space 4}"
    local delta           = "Delta" 
    local delta_abbrev    = "D"
    local delta_lower     = "Dl"
    local delta_upper     = "Du"
    local delta_inflate   = ""
    local epsilon         = "epsilon"
    local epsilon_lower   = "el"
    local epsilon_upper   = "eu"
    local epsilon_inflate = ""
    local lessthan        = "<="
    local greaterthan     = ">="
    local notequal        = "!="
    local caret           = "^"
    local times           = "*"
    local apostrophe      = "'"
    }
   else {
    version 14.0, missing
    * Create Unicode output constants
    local alpha_out       = uchar(0945)
    local theta           = uchar(0952)
    local theta_inflate   = "    "
    local delta           = uchar(0916)
    local delta_abbrev    = "`delta'"
    local delta_lower     = uchar(0916)+uchar(0108)
    local delta_upper     = uchar(0916)+uchar(0117)
    local delta_inflate   = "    "
    local epsilon         = uchar(0949)
    local epsilon_lower   = uchar(0949)+uchar(0108)
    local epsilon_upper   = uchar(0949)+uchar(0117)
    local epsilon_inflate = "      "
    local lessthan        = uchar(8804)
    local greaterthan     = uchar(8805)
    local notequal        = uchar(8800)
    local caret           = uchar(0710)
    local times           = uchar(0215)
    local apostrophe      = uchar(8217)
    }


* a little parsing
 gettoken 1 0 : 0 , parse(" ,")
 gettoken 2 0 : 0 , parse(" ,")
 gettoken 3 0 : 0 , parse(" ,")
 gettoken 4 0 : 0 , parse(" ,")
 gettoken 5 : 0 , parse(" ,")
 if "`5'"=="" | "`5'"=="," {
   local twosample = 0
   }
 if "`5'"!="" & "`5'"!="," {
   local twosample = 1
   gettoken 5 0 : 0 , parse(" ,")
   gettoken 6 0 : 0 , parse(" ,")
   }

 syntax [, EQVType(string) EQVLevel(real 1) UPPEReqvlevel(real 0) UNEqual /*
 */     Welch Alpha(real 0.05) Xname(string) Yname(string) RELevance]

 quietly {

* Validate eqvtype
 if lower("`eqvtype'") == "" {
   local eqvtype = "delta"
   }

 if !(lower("`eqvtype'") == "delta" | lower("`eqvtype'") == "epsilon") {
   noi: di as err "option eqvtype() must be either delta or epsilon"
   exit 198
   }
  if lower("`eqvtype'") == "delta" {
    local eqvtypeout = "`delta'"
    }
  if lower("`eqvtype'") == "epsilon" {
    local eqvtypeout = "`epsilon'"
    }

* Validate eqvlevel
 if (lower("`eqvtype'") == "delta") & (`eqvlevel' == 1 & `uppereqvlevel'==0) {
   local eqvlevel = 1
   }

 if (lower("`eqvtype'") == "epsilon") & (`eqvlevel' == 1 & `uppereqvlevel'==0) {
   local eqvlevel = 2
   }

 if (lower("`eqvtype'") == "delta" || lower("`eqvtype'") == "epsilon") & (`eqvlevel' <= 0 & `uppereqvlevel' != abs(`eqvlevel')) {
   noi: di as err "option eqvlevel() incorrectly specified" _newline "the tolerance must be a positive real value"
   exit 198
   }

* Validate uppereqvlevel
 if (`uppereqvlevel'<0) {
   noi: di as err "option uppereqvlevel() must be a positive real value"
   exit 198
   }
 
 if (`uppereqvlevel'==0 | `uppereqvlevel' == abs(`eqvlevel')) {
   local upper = abs(`eqvlevel')
   local lower = abs(`eqvlevel')
   }

 if (`uppereqvlevel'>0) {
   local upper = abs(`uppereqvlevel')
   local lower = abs(`eqvlevel')
   }
   
* Validate unequal variances against Welch
 if "`welch'" != "" & "`unequal'" == "" {
   noi: di _newline as res "welch option specified, proceeding by assuming unequal variances"
   local unequal = "unequal"
   }

* Invalidate options specified with one-sample test
 if `twosample'==0 {
   if "`welch'" != "" {
     noi: di as err "welch option invalid for one-sample tests"
     exit 198
     } 
   if "`unequal'" != "" {
     noi: di as err "unequal option invalid for one-sample tests"
     exit 198
     } 
   }

* Validate alpha
  if (`alpha' < 0 | `alpha' > 1) {
    noi: di as err "option alpha() must be between 0 and 1 inclusive"
    exit 198
    }
  local level : di %3.2f (100*(1-`alpha'))


*******************************************************************************
* The business starts here                                                    *
*******************************************************************************

**********
* conduct the positivist t test for mean difference
 local PositivistConclusion = "Reject"
 local NegativistConclusion = "Reject"

**********
* one-sample t test for mean equivalence
  if `twosample'==0 {
* tostti #obs #mean #sd #val 
    local n1 = `1'
    local m1 = `2'
    local s1 = `3'
    local mu = `4'
    local df = `n1' - 1
    if `"`xname'"'!="" {
      local name1 = "`xname'"
      }
     else {
      local name1 = "x"	
      }
    local c1 = 49 - length("`name1'")
    if "`relevance'" != "" {
      local t_pos = (`m1'-`mu')/(`s1'/sqrt(`n1'))
      local p_pos = 2 * ttail(`df',abs(`t_pos'))    
      local p_pos_display as txt "= " as res %6.4f `p_pos'
      if `p_pos' < 0.0001 {
        local p_pos_display as txt "< " as res "0.0001"
        }
      if `p_pos' > 0.9999 {
        local p_pos_display as txt "> " as res "0.9999"
        }
      if `p_pos' > `alpha' {
        local PositivistConclusion = "Fail to reject"
        }
      noi: di as txt _newline "One-sample relevance t test of means" _newline
      noi: di as txt "One-sample t test for mean difference"
      noi: _tostttesti1 `name1' `n1' `m1' `s1'/sqrt(`n1') `alpha' 
      noi: di as txt "{hline 9}{c BT}{hline 70}"
      noi: di as txt _col(3) "`theta_inflate'`theta'`caret' = mean(`name1') - " as res `mu' as txt " = " as res %-8.4g `m1'-`mu'  as txt _col(70) "t = " as res %7.0g `t_pos'
      noi: di as txt _col(4) "Ho: `theta' = 0" _col (69) "df =" as res %8.0g `df' _newline
      noi: di as txt _col(37) "Ha: `theta' `notequal' 0"
      noi: di as txt _col(31) "Pr(|T| > |t|) " `p_pos_display' _newline _newline
      }
    if lower("`eqvtype'") == "delta" {
      local t1 = (`upper' - (`m1' - `mu')) / (`s1' / sqrt(`n1'))
      local t2 = ((`m1' - `mu')+`lower') / (`s1' / sqrt(`n1'))
      }
    if lower("`eqvtype'") == "epsilon" {
      local t1 = `upper' - ( ((`m1' - `mu')) / (`s1' / sqrt(`n1')) )
      local t2 = ( ((`m1' - `mu')) / (`s1' / sqrt(`n1')) ) + `lower'
      }
    local p1 = ttail(`df',`t1')
    local p2 = ttail(`df',`t2')
    local p1_display as txt "= " as res %6.4f `p1'
    if `p1' < 0.0001 {
      local p1_display as txt "< " as res "0.0001"
      }
    if `p1' > 0.9999 {
      local p1_display as txt "> " as res "0.9999"
      }
    local p2_display as txt "= " as res %6.4f `p2'
    if `p2' < 0.0001 {
      local p2_display as txt "< " as res "0.0001"
      }
    if `p2' > 0.9999 {
      local p2_display as txt "> " as res "0.9999"
      }
    if (`p1' > `alpha' | `p2' > `alpha') {
      local NegativistConclusion = "Fail to reject"
      }
    noi: di as text _newline "One-sample t test for mean equivalence"
    if (mod(`level'*10, 1) == 0.0) {
      local level: di %3.1f `level'
      }
    if (mod(`level', 1) == 0.0) {
      local level: di %3.0f `level'
      }
    noi: _tostttesti1 `name1' `n1' `m1' `s1'/sqrt(`n1') `alpha' 
    if lower("`eqvtype'") == "delta" {
      noi: di as txt "{hline 9}{c +}{hline 70}"
      if (`upper' == `lower') {
        noi: _tostttesti2 "`eqvtypeout'-`theta'`caret'" `n1' `upper'-(`m1'-`mu') `s1'/sqrt(`n1') `df' `alpha'
        noi: _tostttesti2 "`theta'`caret'+`eqvtypeout'" `n1' (`m1'-`mu')+`lower' `s1'/sqrt(`n1') `df' `alpha'
        }
      if (`upper' != `lower') {
        noi: _tostttesti2 "`delta_abbrev'u-`theta'`caret'" `n1' `upper'-(`m1'-`mu') `s1'/sqrt(`n1') `df' `alpha'
        noi: _tostttesti2 "`theta'`caret'-`delta_abbrev'l" `n1' (`m1'-`mu')+`lower' `s1'/sqrt(`n1') `df' `alpha'
       }
     }
    noi: di as txt "{hline 9}{c BT}{hline 70}"
    noi: di as txt _col(3) "`theta_inflate'`theta'`caret' = mean(`name1') - "  as res `mu' as txt " = " as res %-8.4g `m1'-`mu'
    local neg_t_stats_out noi: di as txt _col(9) "t1 = " as res %-8.4g `t1' as txt _col(38) "t2 = " as res %-8.4g `t2' _newline
    local neg_p_vals_out noi: di as txt _col(4) "Pr(T > t1) " `p1_display' _col(32) as txt " Pr(T > t2) " `p2_display'
    if lower("`eqvtype'") == "delta" {
      if (`upper' == `lower') {
        noi: di as txt _col(4) "`delta_inflate'`eqvtypeout' = " as res %-8.4f `lower' as txt _col(21) "`eqvtypeout' expressed in same units as `name1'"
        }
      if (`upper' != `lower') {
        noi: di as txt _col(3) "`delta_inflate'`eqvtypeout'l = " as res %-8.4f -1*`lower' as txt _col(21) "`eqvtypeout'l expressed in same units as `name1'"
        noi: di as txt _col(3) "`delta_inflate'`eqvtypeout'u =  " as res %-8.4f `upper' as txt _col(21) "`eqvtypeout'u expressed in same units as `name1'"
        }
      noi: di as txt _col(7) "df = " as res %-8.0g `df' as txt _col(21) "using `n1' - 1"
      local criticalvalue = (`s1'/sqrt(`n1'))*invttail(`df',`alpha')
      if abs(`criticalvalue') < 1 {
        local criticalvalue_display: di "0" %-6.4g `criticalvalue'        
        }
       else {
        local criticalvalue_display: di %-6.4g `criticalvalue'        
        }
      if (`upper' == `lower' & `lower' <= `criticalvalue') {
        noi: di _newline as res " Impossible to reject any Ho if `eqvtypeout' `lessthan' t-crit`times's.e. (" trim("`criticalvalue_display'") "). See{help tostt##mineqvlevel: help tostti}." _newline
        }
      if (`upper' != `lower' & `lower' <= `criticalvalue') {
        noi: di _newline as res " Impossible to reject any Ho if |`eqvtypeout'l| `lessthan' t-crit`times's.e. (" trim("`criticalvalue_display'") "). See{help tostt##mineqvlevel: help tostti}." _newline
        }
      if (`upper' != `lower' & `upper' <= `criticalvalue') {
        noi: di _newline as res " Impossible to reject any Ho if `eqvtypeout'u `lessthan' t-crit`times's.e. (" trim("`criticalvalue_display'") "). See{help tostt##mineqvlevel: help tostti}." _newline
        }
      if (`upper' == `lower') {
        noi: di _newline as txt "Ho: |`theta'| `greaterthan' `eqvtypeout':" _newline 
        `neg_t_stats_out'
        noi: di as txt _col(4) "Ho1: `eqvtypeout'-`theta' `lessthan' 0" _col(33) "Ho2: `theta'+`eqvtypeout' `lessthan' 0"
        noi: di as txt _col(4) "Ha1: `eqvtypeout'-`theta' > 0" _col(33) "Ha2: `theta'+`eqvtypeout' > 0"
        `neg_p_vals_out'
        }
      if (`upper' != `lower') {
        noi: di _newline as txt "Ho: `theta' `lessthan' `eqvtypeout'l, or `theta' `greaterthan' `eqvtypeout'u:" _newline 
        `neg_t_stats_out'
        noi: di as txt _col(4) "Ho1: `eqvtypeout'u-`theta' `lessthan' 0" _col(33) "Ho2: `theta'-`eqvtypeout'l `lessthan' 0"
        noi: di as txt _col(4) "Ha1: `eqvtypeout'u-`theta' > 0" _col(33) "Ha2: `theta'-`eqvtypeout'l > 0"
        `neg_p_vals_out'
        }
      }
    if lower("`eqvtype'") == "epsilon" {
      if (`upper' == `lower') {
        noi: di as txt _col(2) "`epsilon_inflate'`eqvtypeout' = " as res %-8.4f `lower' as txt _col(21) "`eqvtypeout' expressed in units of the t distribution"
        }
      if (`upper' != `lower') {
        noi: di as txt "`epsilon_inflate'`eqvtypeout'l = " as res %-8.4f -1*`lower' as txt _col(21) "`eqvtypeout'l expressed in units of the t distribution"
        noi: di as txt "`epsilon_inflate'`eqvtypeout'u = " as res %-8.4f `upper' as txt _col(21) "`eqvtypeout'u expressed in units of the t distribution"
        }
      noi: di as txt _col(7) "df = " as res %-8.0g `df' as txt _col(21) "using `n1' - 1"
      local criticalvalue = invttail(`df',`alpha')
      local criticalvalue_display: di %-5.3f `criticalvalue'
      if (`upper' == `lower' & `lower' <= `criticalvalue') {
        noi: di _newline as res " Impossible to reject any Ho if `eqvtypeout' `lessthan' t-crit (" trim("`criticalvalue_display'") "). See{help tostt##mineqvlevel: help tostti}." _newline
        }
      if (`upper' != `lower' & `lower' <= `criticalvalue') {
        noi: di _newline as res " Impossible to reject any Ho if |`eqvtypeout'l| `lessthan' t-crit (" trim("`criticalvalue_display'") "). See{help tostt##mineqvlevel: help tostti}." _newline
        }
      if (`upper' != `lower' & `upper' <= `criticalvalue') {
        noi: di _newline as res " Impossible to reject any Ho if `eqvtypeout'u `lessthan' t-crit (" trim("`criticalvalue_display'") "). See{help tostt##mineqvlevel: help tostti}." _newline
        }
      if (`upper' == `lower') {
        noi: di _newline as txt "Ho: |T| `greaterthan' `eqvtypeout':" _newline 
        `neg_t_stats_out'
        noi: di as txt _col(4) "Ho1: `eqvtypeout'-T `lessthan' 0" _col(33) "Ho2: T+`eqvtypeout' `lessthan' 0"
        noi: di as txt _col(4) "Ha1: `eqvtypeout'-T > 0" _col(33) "Ha2: T+`eqvtypeout' > 0"
        `neg_p_vals_out'
        }
      if (`upper' != `lower') {
        noi: di _newline as txt "Ho: T `lessthan' `eqvtypeout'l, or T `greaterthan' `eqvtypeout'u:" _newline 
        `neg_t_stats_out'
        noi: di as txt _col(4) "Ho1: `eqvtypeout'u-T `lessthan' 0" _col(33) "Ho2: T-`eqvtypeout'l `lessthan' 0"
        noi: di as txt _col(4) "Ha1: `eqvtypeout'u-T > 0" _col(33) "Ha2: T-`eqvtypeout'l > 0"
        `neg_p_vals_out'
        }
      }
    }


**********
* two-sample unpaired t test for mean equivalence
 if `twosample'==1 {
    local n1 = `1'
    local m1 = `2'
    local s1 = `3'
    local v1 = `s1'^2
    local n2 = `4'
    local m2 = `5'
    local s2 = `6'
    local v2 = `s2'^2
    local s = sqrt(((`s1'^2)/`n1') + ((`s2'^2)/`n2'))
    local n = `n1'+`n2'
    local m = ((`m1'*`n1') + (`m2'*`n2'))/`n'
    local s_combined = sqrt(((`s1'^2)*(`n1'-1) + ((`m' - `m1')^2)*`n1' + ((`m'-`m2')^2)*`n2' + (`s2'^2)*(`n2'-1))/(`n'-1))
    if "`unequal'" == "" {
      local variance_assumption = "with equal variances"
      local df = `n1' + `n2' - 2
      local df_explain = "using `n1' + `n2' - 2"
      local df_display = "degrees of freedom = (`n' - 2)"
      local df_pad = 80  - ulength("`df_display'") - ulength(" df =") - 4
      local se = sqrt((`s1' * (`n1' - 1)) + (`s2' * (`n2' - 1)) * (1 / `df') * (1/`n1' + 1/`n2'))
      }
    if "`unequal'" != "" {
      local variance_assumption = "with unequal variances"
      if "`welch'" == "" {
        local df = (((`v1'/`n1') + (`v2'/`n2'))^2)/((((`v1'/`n1')^2)/(`n1'-1)) + (((`v2'/`n2')^2)/(`n2'-1)))
        local df_explain = "using Satterthwaite`apostrophe's formula"
        local df_display = "Satterthwaite`apostrophe's degrees of freedom"
        local df_pad = 80  - ulength("`df_display'") - ulength(" df =") - 4
        }
      if "`welch'" != "" {
        local df = -2 + (((`v1'/`n1') + (`v2'/`n2'))^2)/((((`v1'/`n1')^2)/(`n1'+1)) + (((`n2'/`n2')^2)/(`n2'+1)))
        local df_explain = "using Welch`apostrophe's formula"
        local df_display = "Welch`apostrophe's degrees of freedom"
        local df_pad = 80 - ulength("`df_display'") - ulength(" df =") - 4
        }
      local se = sqrt((`s1'^2) / `n1' + (`s2'^2) / `n2')
      }
    if `"`xname'"'!="" {
      local name1 = "`xname'"
      }
     else {
      local name1 = "x"	
      }
    if `"`yname'"'!="" {
      local name2 = "`yname'"
      }
     else {
      local name2 = "y"	
      }
    local c1 = 49 - length("`name1'")
    local c2 = 49 - length("`name2'")
    if "`relevance'" != "" {
      local t_pos = (`m1'-`m2')/`s'
      local p_pos = 2 * ttail(`df',abs(`t_pos'))
      local p_pos_display as txt "= " as res %6.4f `p_pos'
      if `p_pos' < 0.0001 {
        local p_pos_display as txt "< " as res "0.0001"
        }
      if `p_pos' > 0.9999 {
        local p_pos_display as txt "> " as res "0.9999"
        }
      if `p_pos' > `alpha' {
        local PositivistConclusion = "Fail to reject"
        }
      noi: di as txt _newline "Two-sample unpaired relevance t test of means" _newline
      noi: di as txt `"Two-sample unpaired t test for mean difference `variance_assumption'"'
      noi: _tostttesti1 `name1' `n1' `m1' `s1'/sqrt(`n1') `alpha' 
      noi: _tostttesti3 `name2' `n2' `m2' `s2'/sqrt(`n2') `alpha' 
      noi: di as txt "{hline 9}{c +}{hline 70}"
      noi: _tostttesti3 "Combined" `n' `m' `s_combined'/sqrt(`n') `alpha'
      noi: di as txt "{hline 9}{c +}{hline 70}"
      noi: _tostttesti2 "`theta'`caret'" `n' `m1'-`m2' `s' `df' `alpha'         
      noi: di as txt "{hline 9}{c BT}{hline 70}"
      noi: di as txt _col(3) "`theta_inflate'`theta'`caret' = mean(`name1') - mean(`name2') = " as res %-8.4g `m1' - `m2' as txt  _col(70) "t = " as res %-7.4f `t_pos'
      noi: di as txt _col(4) "Ho: `theta' = 0" _col (`df_pad') "`df_display' =" as res %8.0g `df' _newline
      noi: di as txt _col(37) "Ha: `theta' `notequal' 0"
      noi: di as txt _col(31) "Pr(|T| > |t|) " `p_pos_display' _newline
      }
    if lower("`eqvtype'") == "delta" {
      local t1 = (`upper' - (`m1' - `m2')) / `s'
      local t2 = ((`m1' - `m2') + `lower') / `s'
      }
    if lower("`eqvtype'") == "epsilon" {
      local t1 = `upper' - ( ( (`m1' - `m2'))/`s' )
      local t2 = ( ((`m1' - `m2'))/`s' ) + `lower'
      }
    local p1 = ttail(`df',`t1')
    local p2 = ttail(`df',`t2')
    local p1_display as txt "= " as res %6.4f `p1'
    if `p1' < 0.0001 {
      local p1_display as txt "< " as res "0.0001"
      }
    if `p1' > 0.9999 {
      local p1_display as txt "> " as res "0.9999"
      }
    local p2_display as txt "= " as res %6.4f `p2'
    if `p2' < 0.0001 {
      local p2_display as txt "< " as res "0.0001"
      }
    if `p2' > 0.9999 {
      local p2_display as txt "> " as res "0.9999"
      }
    if (`p1' > `alpha' | `p2' > `alpha') {
      local NegativistConclusion = "Fail to reject"
      }
    if ("`unequal'" == "" & "`welch'" == "") {
     noi: di as text _newline "Two-sample unpaired t test for mean equivalence with equal variances"
     }
    if ("`unequal'" != "" | "`welch'" != ""){
      noi: di as text _newline "Two-sample unpaired t test for mean equivalence with unequal variances"
      }
    if "`xname'" == "" {
      local xname = "x"
      }
    if "`yname'" == "" {
      local yname = "y"
      }
    if (mod(`level'*10, 1) == 0.0) {
      local level: di %3.1f `level'
      }
    if (mod(`level', 1) == 0.0) {
      local level: di %3.0f `level'
      }
    noi: _tostttesti1 `name1' `n1' `m1' `s1'/sqrt(`n1') `alpha' 
    noi: _tostttesti3 `name2' `n2' `m2' `s2'/sqrt(`n2') `alpha' 
    noi: di as txt "{hline 9}{c +}{hline 70}"
    noi: _tostttesti2 "`theta'`caret'" `n' `m1'-`m2' `s' `df' `alpha'         
    if lower("`eqvtype'") == "delta" {
      noi: di as txt "{hline 9}{c +}{hline 70}"
      if (`upper' == `lower') {
        noi: _tostttesti2 "`delta_abbrev'-`theta'`caret'" `n' `upper'-(`m1'-`m2') `s' `df' `alpha'
        noi: _tostttesti2 "`theta'`caret'+`delta_abbrev'" `n' (`m1'-`m2')+`lower' `s' `df' `alpha'
        }
      if (`upper' != `lower') {
        noi: _tostttesti2 "`delta_abbrev'u-`theta'`caret'" `n1' `upper'-(`m1'-`m2') `s' `df' `alpha'
        noi: _tostttesti2 "`theta'`caret'-`delta_abbrev'l" `n1' (`m1'-`m2')+`lower' `s' `df' `alpha'
        }
     }
    if (`p1' > `alpha' | `p2' > `alpha') {
      local NegativistConclusion = "Fail to reject"
      }
    noi: di as txt "{hline 9}{c BT}{hline 70}"
    noi: di as txt _col(3) "`theta_inflate'`theta'`caret' = mean(`name1') - mean(`name2') = " as res %-8.4g `m1' - `m2'
    if lower("`eqvtype'") == "delta" {
      if (`upper' == `lower') {
        noi: di as txt _col(4) "`delta_inflate'`delta' = " as res %-8.4f `lower' as txt _col(21) "`eqvtypeout' expressed in same units as `name1' and `name2'"
        }
      if (`upper' != `lower') {
        noi: di as txt _col(7) "`delta_abbrev'l = " as res %-8.4f -1*`lower' as txt _col(21) "`eqvtypeout'l expressed in same units as `name1' and `name2'"
        noi: di as txt _col(7) "`delta_abbrev'u =  " as res %-8.4f `upper' as txt _col(21) "`eqvtypeout'u expressed in same units as `name1' and `name2'"
        }
      if "`welch'" == "" & "`unequal'" != "" {
        noi: di as text _col(7) "df = " as res %-8.0g `df' as text " using Satterthwaite's formula"
        }
      if "`welch'" != "" & "`unequal'" != "" {
        noi: di as text _col(7) "df = " as res %-8.0g `df' as text " using Welch's formula"
        }
      if  "`unequal'" == "" {
        noi: di as text _col(7) "df = " as res %-8.0g `df' as text " using `n1' + `n2' - 2"
        }
      local criticalvalue = `s'*invttail(`df',`alpha')
      if abs(`criticalvalue') < 1 {
        local criticalvalue_display: di "0" %-6.4g `criticalvalue'
        }
       else {
        local criticalvalue_display: di %-6.4g `criticalvalue'
        }
      if (`upper' == `lower' & `lower' <= `criticalvalue') {
        noi: di _newline as res " Impossible to reject any Ho if `eqvtypeout' `lessthan' t-crit`times's.e. (" trim("`criticalvalue_display'") "). See{help tostt##mineqvlevel: help tostt}."
        }
      if (`upper' != `lower' & `lower' <= `criticalvalue') {
        noi: di _newline as res " Impossible to reject any Ho if |`eqvtypeout'l| `lessthan' t-crit`times's.e. (" trim("`criticalvalue_display'") "). See{help tostt##mineqvlevel: help tostt}."
        }
      if (`upper' != `lower' & `upper' <= `criticalvalue') {
        noi: di _newline as res " Impossible to reject any Ho if `eqvtypeout'u `lessthan' t-crit`times's.e. (" trim("`criticalvalue_display'") "). See{help tostt##mineqvlevel: help tostt}."
        }
      }
    if lower("`eqvtype'") == "epsilon" {
      if (`upper' == `lower') {
        noi: di as txt _col(2) "`epsilon_inflate'`eqvtypeout' = " as res %-8.4f `lower' as txt _col(21) "`eqvtypeout' expressed in units of the t distribution"
        }
      if (`upper' != `lower') {
        noi: di as txt "`epsilon_inflate'`eqvtypeout'l = " as res %-8.4f -1*`lower' as txt _col(21) "`eqvtypeout'l expressed in units of the t distribution"
        noi: di as txt "`epsilon_inflate'`eqvtypeout'u =  " as res %-8.4f `upper' as txt _col(21) "`eqvtypeout'u expressed in units of the t distribution"
        }
      if "`welch'" == "" & "`unequal'" != "" {
        noi: di as text _col(7) "df = " as res %-8.0g `df' as text " using Satterthwaite's formula"
        }
      if "`welch'" != "" & "`unequal'" != "" {
        noi: di as text _col(7) "df = " as res %-8.0g `df' as text " using Welch's formula"
        }
      if  "`unequal'" == "" {
        noi: di as text _col(7) "df = " as res %-8.0g `df' as text " using `n1' + `n2' - 2"
        }
      local criticalvalue = invttail(`df',`alpha')
      local criticalvalue_display: di %-5.3f `criticalvalue'
      if (`upper' == `lower' & `lower' <= `criticalvalue') {
        noi: di _newline as res " Impossible to reject any Ho if `eqvtypeout' `lessthan' t-crit (" trim("`criticalvalue_display'") "). See{help tostt##mineqvlevel: help tostt}."
        }
      if (`upper' != `lower' & `lower' <= `criticalvalue') {
        noi: di _newline as res " Impossible to reject any Ho if |`eqvtypeout'l| `lessthan' t-crit (" trim("`criticalvalue_display'") "). See{help tostt##mineqvlevel: help tostt}."
        }
      if (`upper' != `lower' & `upper' <= `criticalvalue') {
        noi: di _newline as res " Impossible to reject any Ho if `eqvtypeout'u `lessthan' t-crit (" trim("`criticalvalue_display'") "). See{help tostt##mineqvlevel: help tostt}."
        }
      }
    local neg_t_stats_out noi: di as txt _col(9) "t1 = " as res %-8.4g `t1' as txt _col(38) "t2 = " as res %-8.4g `t2' _newline
    local neg_p_vals_out noi: di as txt _col(4) "Pr(T > t1) " `p1_display' _col(32) as txt " Pr(T > t2) " `p2_display'
    if lower("`eqvtype'") == "delta" {
      if (`upper' == `lower') {
        noi: di _newline as txt "Ho: |`theta'| `greaterthan' `eqvtypeout':" _newline 
        `neg_t_stats_out'
        noi: di as txt _col(4) "Ho1: `eqvtypeout'-`theta' `lessthan' 0" _col(33) "Ho2: `theta'+`eqvtypeout' `lessthan' 0"
        noi: di as txt _col(4) "Ha1: `eqvtypeout'-`theta' > 0" _col(33) "Ha2: `theta'+`eqvtypeout' > 0"
        `neg_p_vals_out'
        }
      if (`upper' != `lower') {
        noi: di _newline as txt "Ho: `theta' `lessthan' `eqvtypeout'l, or diff `greaterthan' `eqvtypeout'u:" _newline 
        `neg_t_stats_out'
        noi: di as txt _col(4) "Ho1: `eqvtypeout'u-diff `lessthan' 0" _col(33) "Ho2: diff-`eqvtypeout'l `lessthan' 0"
        noi: di as txt _col(4) "Ha1: `eqvtypeout'u-diff > 0" _col(33) "Ha2: diff-`eqvtypeout'l > 0"
        `neg_p_vals_out'
        }
      }
    if lower("`eqvtype'") == "epsilon" {
      if (`upper' == `lower') {
        noi: di _newline as txt "Ho: |T| `greaterthan' `eqvtypeout':" _newline 
        `neg_t_stats_out'
        noi: di as txt _col(4) "Ho1: `eqvtypeout'-T `lessthan' 0" _col(33) "Ho2: T+`eqvtypeout' `lessthan' 0"
        noi: di as txt _col(4) "Ha1: `eqvtypeout'-T > 0" _col(33) "Ha2: T+`eqvtypeout' > 0"
        `neg_p_vals_out'
        }
      if (`upper' != `lower') {
        noi: di _newline as txt "Ho: T `lessthan' `eqvtypeout'l, or T `greaterthan' `eqvtypeout'u:" _newline 
        `neg_t_stats_out'
        noi: di as txt _col(4) "Ho1: `eqvtypeout'u-T `lessthan' 0" _col(33) "Ho2: T-`eqvtypeout'l `lessthan' 0"
        noi: di as txt _col(4) "Ha1: `eqvtypeout'u-T > 0" _col(33) "Ha2: T-`eqvtypeout'l > 0"
        `neg_p_vals_out'
        }
      }
    }

* Output combined tests results if relevance test is requested
 if "`relevance'" != "" {
   * Format alpha to remove trailing zeros
   if (mod(`alpha'*1000, 1) == 0.0) {
     local alpha: di %6.3f `alpha'
     }
   if (mod(`alpha'*100, 1) == 0.0) {
     local alpha: di %5.2f `alpha'
     }
   if (mod(`alpha'*10, 1) == 0.0) {
     local alpha: di %4.1f `alpha'
     }
   if (mod(`alpha', 1) == 0.0) {
     local alpha: di %4.0f `alpha'
     }

   * Format Delta or epsilon to remove trailing zeros
   if (mod(`lower'*1000, 1) == 0.0) {
     local lower: di %6.3f `lower'
     }
   if (mod(`lower'*100, 1) == 0.0) {
     local lower: di %5.2f `lower'
     }
   if (mod(`lower'*10, 1) == 0.0) {
     local lower: di %4.1f `lower'
     }
   if (mod(`lower', 1) == 0.0) {
     local lower: di %4.0f `lower'
     }
   if (`upper' != `lower') {
     if (mod(`upper'*1000, 1) == 0.0) {
       local upper: di %6.3f `upper'
       }
     if (mod(`upper'*100, 1) == 0.0) {
       local upper: di %5.2f `upper'
       }
     if (mod(`upper'*10, 1) == 0.0) {
       local upper: di %4.1f `upper'
       }
     if (mod(`upper', 1) == 0.0) {
       local upper: di %4.0f `upper'
       }
     }
    if (`upper' == `lower') {
      if lower("`eqvtype'") == "delta" {
        if (`lower' < 1) {
          noi: di _newline _newline as txt "Relevance test conclusion for `alpha_out' = " as res "0" `alpha' as txt ", and `eqvtypeout' = " as res "0" `lower' as txt ":"
          }
         else {
          noi: di _newline _newline as txt "Relevance test conclusion for `alpha_out' = " as res "0" `alpha' as txt ", and `eqvtypeout' = " as res `lower' as txt ":"
          }
        }
      if lower("`eqvtype'") == "epsilon" {
        noi: di _newline _newline as txt "Relevance test conclusion for `alpha_out' = " as res "0" `alpha' as txt ", and `eqvtypeout' = " as res `lower' as txt ":"
        }
      }
    if (`upper' != `lower') {
      if lower("`eqvtype'") == "delta" {
        if (`lower' < 1 & `upper' < 1) {
          noi: di _newline _newline as txt "Relevance test conclusion for `alpha_out' = " as res "0" `alpha' as txt ", `eqvtypeout'l = " as res "-0" `lower' as txt ", and `eqvtypeout'u = " as res "0" `upper' as txt ":"
          }
        if (`lower' >= 1 & `upper' < 1) {
          noi: di _newline _newline as txt "Relevance test conclusion for `alpha_out' = " as res "0" `alpha' as txt ", `eqvtypeout'l = " as res -1*`lower' as txt  ", and `eqvtypeout'u = " as res "0" `upper' as txt ":"
          }
        if (`lower' < 1 & `upper' >= 1) {
          noi: di _newline _newline as txt "Relevance test conclusion for `alpha_out' = " as res "0" `alpha' as txt ", `eqvtypeout'l = " as res "-0" `lower' as txt  ", and `eqvtypeout'u = " as res `upper' as txt ":"
          }
        if (`lower' >= 1 & `upper' >= 1) {
          noi: di _newline _newline as txt "Relevance test conclusion for `alpha_out' = " as res "0" `alpha' as txt ", `eqvtypeout'l = " as res -1*`lower' as txt  ", and `eqvtypeout'u = " as res `upper' as txt ":"
          }
        }
      if lower("`eqvtype'") == "epsilon" {
        noi: di _newline _newline as txt "Relevance test conclusion for `alpha_out' = " as res "0" `alpha' as txt ", `eqvtypeout'l = " as res -1*`lower' as txt  ", and `eqvtypeout'u = " as res `upper' as txt ":"
        }
      }
   noi: di as txt "  Ho test for difference:  " as res "`PositivistConclusion'" 
   noi: di as txt "  Ho test for equivalence: " as res "`NegativistConclusion'" 
   if "`PositivistConclusion'" == "Reject" & "`NegativistConclusion'" == "Reject" {
     local RelevanceTestConclusion = "Trivial difference (overpowered test)"
     }
   if "`PositivistConclusion'" == "Reject" & "`NegativistConclusion'" == "Fail to reject" {
     local RelevanceTestConclusion = "Relevant difference"
     }
   if "`PositivistConclusion'" == "Fail to reject" & "`NegativistConclusion'" == "Reject" {
     local RelevanceTestConclusion = "Equivalence"
     }
   if "`PositivistConclusion'" == "Fail to reject" & "`NegativistConclusion'" == "Fail to reject" {
     local RelevanceTestConclusion = "Indeterminate (underpowered test)"
     }
   noi: di _newline as txt "Conclusion from combined tests: " as res "`RelevanceTestConclusion'" 
   }
  * Close quietly block
  }



*******************************************************************************
* Program end. Close up shop and return things.                               *
*******************************************************************************

  if ("`relevance'" != "") {
    return local relevance = "`RelevanceTestConclusion'"
    }
  if (`upper' == `lower') {
    if "`eqvtype'" == "delta" {
      return scalar Delta   = `eqvlevel'
      }
    if "`eqvtype'" == "epsilon" {
      return scalar epsilon   = `eqvlevel'
      }
    }
  if (`upper' != `lower') {
    if "`eqvtype'" == "delta" {
      return scalar Dl   = `lower'
      return scalar Du   = `upper'
      }
    if "`eqvtype'" == "epsilon" {
      return scalar el   = `lower'
      return scalar eu   = `upper'
      }
    }
  return scalar N_1     = `n1'
  return scalar mu_1    = `m1'
  if `twosample'==1 {
    return scalar N_2   = `n2'
    return scalar mu_2  = `m2'
    }
  return scalar df_t    = `df'
  return scalar t1      = `t1'
  return scalar t2      = `t2'
  return scalar p1      = `p1'
  return scalar p2      = `p2'
  if `twosample'==1 {
    return scalar se    = `s'
    }
  else {
   return scalar se     = `s1'/sqrt(`n1') 
   }
  return scalar sd_1    = `s1'
  if `twosample'==1 {
    return scalar sd_2  = `s2'
    }
  end


program define _tostttesti1
  local name = abbrev(`"`1'"', 8)
  local n "`2'"
  local mean "`3'"
  local se "`4'"
  local sd = `se'*sqrt(`n')
  local alpha "`5'"
  local show = "`6'" 
  if `n' == 1 | `n' >= . {
    local se = .
    }
  if "`show'" != "" {
    local t t`'
    }
  local df = `n' - 1
  local level : di %3.2f (100*(1-`alpha'))
  if (mod(`level'*10, 1) == 0.0) {
    local level: di %3.1f `level'
    }
  if (mod(`level', 1) == 0.0) {
    local level: di %2.0f `level'
    }
  local cil `=string(`level')'
  local cil `=length("`cil'")'
  noi: di as txt "{hline 9}{c TT}{hline 70}"
  noi: di as txt "Variable {c |}" _col(18) "Obs" _col(27) "Mean" _col(35) "Std. err." _col(47) "Std. dev."  _col(`=63-`cil'') `"[`=strsubdp("`level'")'% Conf. Interval]"'
  noi: di as txt "{hline 9}{c +}{hline 70}"
  local vval = `alpha'/2
  noi: di in smcl as txt %8s `"`name'"' /*
  */ as txt _col(10) "{c |}" as res /*
  */ _col(12) %9.0g  `n'   /*
  */ _col(22) %9.0g  `mean'     /*
  */ _col(34) %9.0g  `se'     /*
  */ _col(46) %9.0g  `sd'     /*
  */ _col(60) %9.0g  `mean'-invttail(`df',`vval')*`se'   /*
  */ _col(72) %9.0g  `mean'+invttail(`df',`vval')*`se'
  end

program define _tostttesti2
  local name = abbrev(`"`1'"', 8)
  local n "`2'"
  local mean "`3'"
  local se "`4'"
  local df "`5'"
  local alpha "`6'"
  if `n' == 1 | `n' == . {
    local se = .
    }
  local level : di %3.2f (100*(1-`alpha'))
  if (mod(`level'*10, 1) == 0.0) {
    local level: di %3.1f `level'
    }
  if (mod(`level', 1) == 0.0) {
    local level: di %2.0f `level'
    }
  local vval = `alpha'/2
  noi: di in smcl as txt %8s `"`name'"' " {c |}" as res /*
  		*/ _col(22) %9.0g  `mean'   /*
  		*/ _col(34) %9.0g  `se'     /*
  		*/ _col(60) %9.0g  `mean'-invttail(`df',`vval')*`se' /*
  		*/ _col(72) %9.0g  `mean'+invttail(`df',`vval')*`se'
  end

 
program define _tostttesti3
  local name = abbrev(`"`1'"', 8)
  local n "`2'"
  local mean "`3'"
  local se "`4'"
  local sd = `se'*sqrt(`n')
  local alpha "`5'"
  local show = "`6'" 
  if `n' == 1 | `n' >= . {
    local se = .
    }
  if "`show'" != "" {
    local t t`'
    }
  local df = `n' - 1
  local level : di %3.2f (100*(1-`alpha'))
  if (mod(`level'*10, 1) == 0.0) {
    local level: di %3.1f `level'
    }
  if (mod(`level', 1) == 0.0) {
    local level: di %2.0f `level'
    }
  local cil `=string(`level')'
  local cil `=length("`cil'")'
  local vval = `alpha'/2
  noi: di in smcl as txt %8s `"`name'"' /*
  */ as txt _col(10) "{c |}" as res /*
  */ _col(12) %9.0g  `n'   /*
  */ _col(22) %9.0g  `mean'     /*
  */ _col(34) %9.0g  `se'     /*
  */ _col(46) %9.0g  `sd'     /*
  */ _col(60) %9.0g  `mean'-invttail(`df',`vval')*`se'   /*
  */ _col(72) %9.0g  `mean'+invttail(`df',`vval')*`se'
  end

  
program define _tostttesti4
  local name = abbrev(`"`1'"', 8)
  local n "`2'"
  local mean "`3'"
  local se "`4'"
  local alpha "`5'"
  if `n' == 1 | `n' >= . {
    local se = .
    }
  local df = `n' - 1
  local level : di %3.2f (100*(1-`alpha'))
  if (mod(`level'*10, 1) == 0.0) {
    local level: di %3.1f `level'
    }
  if (mod(`level', 1) == 0.0) {
    local level: di %2.0f `level'
    }
  local cil `=string(`level')'
  local cil `=length("`cil'")'
  local vval = `alpha'/2
  noi: di in smcl as txt %8s `"`name'"' /*
  */ as txt _col(10) "{c |}" as res /*
  */ _col(22) %9.0g  `mean'     /*
  */ _col(34) %9.0g  `se'     /*
  */ _col(60) %9.0g  `mean'-invttail(`df',`vval')*`se'   /*
  */ _col(72) %9.0g  `mean'+invttail(`df',`vval')*`se'
  end

program define _tostttesti5
  local name = abbrev(`"`1'"', 8)
  local n "`2'"
  local mean "`3'"
  local se "`4'"
  local sd = `se'*sqrt(`n')
  local alpha "`5'"
  local show = "`6'" 
  if `n' == 1 | `n' >= . {
    local se = .
    }
  if "`show'" != "" {
    local t t`'
    }
  local df = `n' - 1
  local level : di %3.2f (100*(1-`alpha'))
  if (mod(`level'*10, 1) == 0.0) {
    local level: di %3.1f `level'
    }
  if (mod(`level', 1) == 0.0) {
    local level: di %2.0f `level'
    }
  local cil `=string(`level')'
  local cil `=length("`cil'")'
  noi: di as txt "{hline 9}{c TT}{hline 70}"
  noi: di as txt "   Group {c |}" _col(18) "Obs" _col(27) "Mean" _col(35) "Std. err." _col(47) "Std. dev."  _col(`=63-`cil'') `"[`=strsubdp("`level'")'% Conf. Interval]"'
  noi: di as txt "{hline 9}{c +}{hline 70}"
  local vval = `alpha'/2
  noi: di in smcl as txt %8s `"`name'"' /*
  */ as txt _col(10) "{c |}" as res /*
  */ _col(12) %9.0g  `n'   /*
  */ _col(22) %9.0g  `mean'     /*
  */ _col(34) %9.0g  `se'     /*
  */ _col(46) %9.0g  `sd'     /*
  */ _col(60) %9.0g  `mean'-invttail(`df',`vval')*`se'   /*
  */ _col(72) %9.0g  `mean'+invttail(`df',`vval')*`se'
  end
